<?php

add_action('cmb2_init', 'alt_metabox_loja');


/**
 * Define the metabox and field configurations.
 */
function alt_metabox_loja()
{
    $estados = array(
        'AC' => 'Acre',
        'AL' => 'Alagoas',
        'AP' => 'Amapá',
        'AM' => 'Amazonas',
        'BA' => 'Bahia',
        'CE' => 'Ceará',
        'DF' => 'Distrito Federal',
        'ES' => 'Espirito Santo',
        'GO' => 'Goiás',
        'MA' => 'Maranhão',
        'MS' => 'Mato Grosso do Sul',
        'MT' => 'Mato Grosso',
        'MG' => 'Minas Gerais',
        'PA' => 'Pará',
        'PB' => 'Paraíba',
        'PR' => 'Paraná',
        'PE' => 'Pernambuco',
        'PI' => 'Piauí',
        'RJ' => 'Rio de Janeiro',
        'RN' => 'Rio Grande do Norte',
        'RS' => 'Rio Grande do Sul',
        'RO' => 'Rondônia',
        'RR' => 'Roraima',
        'SC' => 'Santa Catarina',
        'SP' => 'São Paulo',
        'SE' => 'Sergipe',
        'TO' => 'Tocantins',
    );

    // Start with an underscore to hide fields from custom fields list
    $prefix = 'alt_';

    /**
     * Initiate the metabox
     */
    $cmb = new_cmb2_box(array(
        'id' => 'field_lojas',
        'title' => 'Lojas',
        'object_types' => array('lojas'), // Post type
        'context' => 'normal',
        'priority' => 'high',
        'show_names' => true,
        'show_in_rest' => false,
    ));

    $cmb->add_field(array(
        'name' => 'Loja / Usuário',
        'desc' => 'Selecionar o usuário da loja. <br><b>*Campo obrigatório</b>',
        'id' => $prefix . 'lojas_usuario',
        'type' => 'select',
        'default' => '1',
        'options_cb' => 'alt_get_usuarios_callback',
        'attributes' => array(
            'required' => 'required', // Adiciona o atributo required para tornar o campo obrigatório
        ),
    ));

    $cmb->add_field(array(
        'name' => 'Tipo de integração',
        'desc' => 'Selecionar a forma que os veículos serão recebidos no site. O padrão é Altimus.',
        'id' => $prefix . 'lojas_tipo_integracao',
        'type' => 'select',
        'default' => 'altimus',
        'options_cb' => 'alt_tipos_integracoes'
    ));

    $cmb->add_field(array(
        'name' => 'URL do JSON',
        'desc' => 'Informe a URL onde o JSON com os veículos está localizado. <br><b>*Campo obrigatório quando tipo de integração for JSON</b>',
        'id' => $prefix . 'lojas_json_url',
        'type' => 'text_url',
        'protocols' => array('http', 'https')
    ));

    $cmb->add_field(
        array(
            'name' => 'Link',
            'desc' => 'Informe o site da empresa. <br><b>*Campo obrigatório</b>',
            'id' => $prefix . 'lojas_link',
            'type' => 'text_url',
            'protocols' => array('http', 'https'),
            'attributes' => array(
                'required' => 'required',
            ),
        )
    );

    $cmb->add_field(array(
        'name' => 'Whatsapp Principal',
        'desc' => 'Utilizar apenas números. exemplo 4788888888. <br><b>*Campo obrigatório</b>',
        'id' => $prefix . 'lojas_whatsapp_principal',
        'default' => '',
        'type' => 'text',
        'attributes' => array(
            'required' => 'required',
        ),
    ));

    $cmb->add_field(array(
        'name' => 'Telefone fixo',
        'desc' => 'Utilizar apenas números. exemplo 4788888888',
        'id' => $prefix . 'lojas_telefone_principal',
        'default' => '',
        'type' => 'text',
    ));

    $cmb->add_field(array(
        'name' => 'E-mail principal',
        'desc' => '',
        'id' => $prefix . 'lojas_email_principal',
        'default' => '',
        'type' => 'text',
    ));

    $cmb->add_field(
        array(
            'name' => 'Frase empresa',
            'desc' => 'Informe uma frase para a empresa',
            'id' => $prefix . 'lojas_frase',
            'type' => 'text',
            'protocols' => array('http', 'https'),

        )
    );

    $cmb->add_field(array(
        'name' => 'Selecione a logo da empresa',
        'desc' => 'Selecione o arquivo ou informe a URL.',
        'id' => $prefix . 'lojas_imagem',
        'type' => 'file',
        'options' => array(
            'url' => true,
        ),
        'text' => array(
            'add_upload_file_text' => 'Adicionar arquivo'
        ),
        'query_args' => array(
            'type' => array(
                'image/gif',
                'image/jpeg',
                'image/png',
                'image/webp',
            ),
        ),
        'preview_size' => 'thumbnail',
    ));

    $cmb->add_field(array(
        'name' => 'CNPJ',
        'desc' => 'Insira apenas números. Exemplo: 00000000000000',
        'id' => $prefix . 'lojas_cnpj_principal',
        'default' => '',
        'type' => 'text',

    ));
    $cmb->add_field(array(
        'name' => 'Endereço',
        'desc' => 'Ex: R.João Pessoa 2008 - Sala 01',
        'default' => '',
        'id' => $prefix . 'lojas_endereco_principal',
        'type' => 'text',
    ));
    $cmb->add_field(array(
        'name' => 'Cidade',
        'desc' => 'Ex: Blumenau',
        'default' => '',
        'id' => $prefix . 'lojas_cidade_principal',
        'type' => 'text',
    ));
    $cmb->add_field(array(
        'name' => 'Estado',
        'id' => $prefix . 'lojas_estado_principal',
        'type' => 'select',
        'default' => 'SC',
        'options' => $estados,
    ));

    $cmb->add_field(array(
        'name' => 'CEP',
        'desc' => 'Somente números Ex: 89040400',
        'default' => '',
        'id' => $prefix . 'lojas_cep',
        'type' => 'text',
    ));

    $cmb->add_field(array(
        'name' => 'URL Google Maps',
        'desc' => 'Acessar o Google maps para obter a URL. Ex: https://goo.gl/maps/BS6nTdNqkBQeoQbJ7',
        'default' => '',
        'id' => $prefix . 'lojas_google_maps',
        'type' => 'text',
    ));
}

// Adiciona o JavaScript para controlar a visibilidade do campo URL JSON
add_action('admin_footer', 'alt_json_visibilidade_url');
function alt_json_visibilidade_url()
{
?>
    <script>
        jQuery(document).ready(function($) {
            var tipoIntegracao = $('#alt_lojas_tipo_integracao');
            var jsonUrlField = $('#alt_lojas_json_url').closest('.cmb-row');

            // Função para controlar a visibilidade
            function toggleJsonUrlField() {
                if (tipoIntegracao.val() === 'json') {
                    jsonUrlField.show();
                    $('#alt_lojas_json_url').prop('required', true);
                } else {
                    jsonUrlField.hide();
                    $('#alt_lojas_json_url').prop('required', false);
                }
            }

            // Executa ao carregar a página
            toggleJsonUrlField();

            // Executa quando o select for alterado
            tipoIntegracao.on('change', toggleJsonUrlField);
        });
    </script>
<?php
}

// Filtro para remover a barra no final da URL
add_filter('cmb2_sanitize_text_url', function ($override_value, $value, $object_id, $field_args) {
    // Verifica se o campo é o 'alt_lojas_link'
    if (isset($field_args['id']) && $field_args['id'] === 'alt_lojas_link') {
        // Remove a barra final, se existir
        return rtrim($value, '/');
    }

    return $value; // Retorna o valor original para outros campos
}, 10, 4);

/**
 * Retorna um array com todas as lojas publicadas, com seus metadados,
 * ordenadas aleatóriamente.
 *
 * @return array Um array com as lojas publicadas.
 */
function alt_get_lojas()
{
    $args = array(
        'post_type' => 'lojas',
        'post_status' => array('publish'),
        'orderby' => 'rand',
    );

    $query = new WP_Query($args);

    $arrLojas = $query->posts;

    foreach ($arrLojas as $key => $loja) {
        $arrLojas[$key]->meta = get_post_meta($loja->ID);
    }

    wp_reset_query();

    return $arrLojas;
}

/**
 * Retorna o título, link, whatsapp e email de uma loja,
 * relacionada a um usuário com o ID informado.
 *
 * @param int $user_id O ID do usuário.
 * @return array Um array com o título, link, whatsapp e email da loja.
 */
function alt_get_loja_veiculo($user_id)
{
    $args = array(
        'post_type' => 'lojas',
        'post_status' => array('publish'),
        'orderBy' => 'date',
        'order' => 'asc',
        'meta_query' => array(
            array(
                'key' => 'alt_lojas_usuario',
                'value' => $user_id,
                'compare' => 'IN',
            ),
        ),
    );

    $query = new WP_Query($args);

    $result = $query->posts;
    $result = $result[0] ?? null;

    if (!$result || !isset($result->ID)) {
        return [];
    }

    return array(
        'title' => $result->post_title,
        'url' => get_post_meta($result->ID, 'alt_lojas_link', true),
        'whatsapp' => get_post_meta($result->ID, 'alt_lojas_whatsapp_principal', true),
        'email' => get_post_meta($result->ID, 'alt_lojas_email_principal', true),
    );
}

/**
 * Retorna um array com os usuários que podem ser selecionados para uma loja.
 * A lista inclui usuários do tipo lojista que não têm vínculo com lojas.
 * Se o parâmetro $post_id for informado, será retornado um array com o
 * usuário da loja e o seu respectivo nome.
 *
 * @param int $post_id O ID do post da loja.
 * @return array Um array com os usuários que podem ser selecionados para a loja.
 */
function alt_get_usuarios($post_id = null)
{
    // return get_userdata($post_id);
    if (is_null($post_id)) {
        global $wpdb;

        // Subconsulta para obter IDs dos usuários com vínculo com lojas
        $subquery = "
            SELECT DISTINCT meta_value 
            FROM $wpdb->postmeta 
            WHERE meta_key = 'alt_lojas_usuario'
        ";

        // Consulta principal para selecionar usuários do tipo Lojista que não têm vínculo com lojas
        $sql = "
            SELECT u.ID, u.display_name
            FROM $wpdb->users u
            INNER JOIN $wpdb->usermeta um
            ON u.ID = um.user_id
            AND um.meta_key = 'wp_capabilities'
            AND um.meta_value LIKE '%lojista%'
            LEFT JOIN $wpdb->postmeta pm
            ON u.ID = pm.meta_value
            AND pm.meta_key = 'alt_lojas_usuario'
            WHERE pm.meta_value IS NULL
            AND u.ID NOT IN ($subquery)
        ";

        $users = $wpdb->get_results($sql);

        // Converte os resultados para um array associativo
        $user_array = array();

        foreach ($users as $user) {
            $user_array[$user->ID] = $user->display_name;
        }

        // Retorna o array
        return $user_array;
    }
    // buscar o usuário do post
    $usuario_loja = alt_get_user_loja($post_id);
    return [
        $usuario_loja->ID => $usuario_loja->display_name
    ];
}

// Define a função de callback para capturar o ID do post e retornar as opções
// Esta função é usada pelo plugin CMB2 para preencher o campo de usuários
// com as opções de usuários que podem ser selecionados
function alt_get_usuarios_callback($field)
{
    $post = get_post($field->object_id);
    $post_id = null;
    if ($post->post_status != 'auto-draft') {
        $post_id = $post->ID;
    }
    $usuarios = alt_get_usuarios($post_id); // Passa o ID do post para a função que retorna as opções
    return $usuarios;
}

/**
 * Retorna os tipos de integração disponíveis para a loja
 *
 * @return array Um array associativo com os tipos de integração disponíveis
 */
function alt_tipos_integracoes()
{
    return array(
        'altimus' => 'Altimus',
        'api' => 'API',
        'json' => 'Leitura JSON'
    );
}

/**
 * Obtém o tipo de integração configurado para uma loja específica.
 * Retorna 'altimus' como padrão se nenhum tipo estiver definido.
 *
 * @param int $loja_id O ID da loja para a qual se deseja obter o tipo de integração.
 * @return string O tipo de integração configurado ou 'altimus' como padrão.
 */
function alt_get_tipo_integracao($loja_id)
{
    $tipo_integracao = get_post_meta($loja_id, 'alt_lojas_tipo_integracao', true);
    return $tipo_integracao ? $tipo_integracao : 'altimus';
}

// Busca o usuário da loja
/**
 * Retorna o objeto WP_User que representa o usuário que
 * está associado à loja com o ID informado
 * 
 * @param int $post_id ID do post da loja
 * @return WP_User
 */
function alt_get_user_loja($post_id)
{
    $user_id = get_post_meta($post_id, 'alt_lojas_usuario', true);
    return get_userdata($user_id);
}

// Filtro para alterar o autor da loja antes de salvar
function alt_filtrar_autor_loja($data, $postarr)
{
    // Verifica se é uma loja
    if ($data['post_type'] !== 'lojas') {
        return $data;
    }

    // Obtém o ID do usuário do campo lojas_usuario
    $user_id = get_post_meta($postarr['ID'], 'alt_lojas_usuario', true);

    // Se estiver criando um novo post, pega o valor do $_POST
    if (empty($user_id) && isset($_POST['alt_lojas_usuario'])) {
        $user_id = $_POST['alt_lojas_usuario'];
    }

    if (!empty($user_id)) {
        $data['post_author'] = $user_id;
    }

    return $data;
}
add_filter('wp_insert_post_data', 'alt_filtrar_autor_loja', 10, 2);

/**
 * Retorna todos os dados de uma loja com base no ID do veículo
 *
 * @param int $post_id ID do post do veículo
 * @return array Array com todos os dados da loja
 */
function alt_get_dados_loja($post_id) {
    // Obtém o autor do veículo
    $veiculo = get_post($post_id);
    if (!$veiculo) {
        return [];
    }
    
    $autor_id = $veiculo->post_author;

    // Busca a loja que tem este usuário
    $args = array(
        'post_type' => 'lojas',
        'post_status' => 'publish',
        'meta_query' => array(
            array(
                'key' => 'alt_lojas_usuario',
                'value' => $autor_id,
                'compare' => '='
            )
        )
    );

    $query = new WP_Query($args);
    
    if (!$query->have_posts()) {
        return [];
    }

    $loja = $query->posts[0];
    $loja_id = $loja->ID;
    
    return [
        'id' => $loja_id,
        'titulo' => $loja->post_title,
        'link' => get_post_meta($loja_id, 'alt_lojas_link', true),
        'whatsapp' => get_post_meta($loja_id, 'alt_lojas_whatsapp_principal', true),
        'telefone' => get_post_meta($loja_id, 'alt_lojas_telefone_principal', true),
        'email' => get_post_meta($loja_id, 'alt_lojas_email_principal', true),
        'frase' => get_post_meta($loja_id, 'alt_lojas_frase', true),
        'logo' => get_post_meta($loja_id, 'alt_lojas_imagem', true),
        'cnpj' => get_post_meta($loja_id, 'alt_lojas_cnpj_principal', true),
        'endereco' => get_post_meta($loja_id, 'alt_lojas_endereco_principal', true),
        'cidade' => get_post_meta($loja_id, 'alt_lojas_cidade_principal', true),
        'estado' => get_post_meta($loja_id, 'alt_lojas_estado_principal', true),
        'cep' => get_post_meta($loja_id, 'alt_lojas_cep', true),
        'maps_url' => get_post_meta($loja_id, 'alt_lojas_google_maps', true),
        'tipo_integracao' => alt_get_tipo_integracao($loja_id)
    ];
}