<?php


/**
 * Funções para gerenciar a integração entre usuários e lojas
 */

/**
 * Processa o upload de uma logo para a loja
 * 
 * @param array $file Arquivo enviado
 * @param int $loja_id ID da loja
 * @return bool Sucesso ou falha
 */
function alt_processar_upload_logo($file, $loja_id) {
    if (!function_exists('wp_handle_upload')) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
    }

    $uploadedfile = $file;
    $upload_overrides = array('test_form' => false);
    $movefile = wp_handle_upload($uploadedfile, $upload_overrides);

    if ($movefile && !isset($movefile['error'])) {
        update_post_meta($loja_id, 'alt_lojas_logo', $movefile['url']);
        return true;
    }
    return false;
}

/**
 * Sincroniza a loja com o usuário sempre que há alteração
 * 
 * @param int $post_id ID do post
 * @param WP_Post $post Objeto do post
 * @return void
 */
function alt_sincronizar_loja_usuario($post_id, $post) {
    // Evitar execução durante edição na página de usuário_loja
    if (isset($_GET['page']) && $_GET['page'] === 'usuario_loja' && isset($_GET['action']) && $_GET['action'] === 'edit') {
        return;
    }
    
    // Verifica se é do tipo loja
    if ($post->post_type !== 'lojas') {
        return;
    }

    // Obtém o ID do usuário vinculado
    $user_id = get_post_meta($post_id, 'alt_lojas_usuario', true);
    if (empty($user_id)) {
        return;
    }

    // Define o autor do post como o usuário vinculado
    if ($post->post_author != $user_id) {
        wp_update_post(array(
            'ID' => $post_id,
            'post_author' => $user_id
        ));
    }
    
    // Atualiza informações do usuário com base na loja
    $user_data = array(
        'ID' => $user_id
    );
    
    // Sincroniza o nome de exibição com o título da loja
    if (!empty($post->post_title)) {
        $user_data['display_name'] = $post->post_title;
    }
    
    // Sincroniza o e-mail
    $email_loja = get_post_meta($post_id, 'alt_lojas_email_principal', true);
    if (!empty($email_loja)) {
        // Verifica se o e-mail já está em uso por outro usuário
        $existing_user = get_user_by('email', $email_loja);
        if (!$existing_user || $existing_user->ID == $user_id) {
            $user_data['user_email'] = $email_loja;
        }
    }
    
    // Atualiza os dados do usuário
    if (count($user_data) > 1) { // Se houver mais que apenas o ID
        wp_update_user($user_data);
    }
}
add_action('save_post', 'alt_sincronizar_loja_usuario', 20, 2);

/**
 * Obtém a loja vinculada ao usuário
 * 
 * @param int $user_id ID do usuário
 * @return WP_Post|false Objeto da loja ou false se não encontrado
 */
function alt_get_loja_do_usuario($user_id) {
    $args = array(
        'post_type' => 'lojas',
        'post_status' => 'publish',
        'posts_per_page' => 1,
        'meta_query' => array(
            array(
                'key' => 'alt_lojas_usuario',
                'value' => $user_id,
                'compare' => '='
            )
        )
    );
    
    $lojas = get_posts($args);
    
    if (!empty($lojas)) {
        return $lojas[0];
    }
    
    return false;
}

/**
 * Vincula um usuário a uma loja
 * 
 * @param int $user_id ID do usuário
 * @param int $loja_id ID da loja
 * @return bool Sucesso ou falha
 */
function alt_vincular_usuario_loja($user_id, $loja_id) {
    $user = get_userdata($user_id);
    $loja = get_post($loja_id);
    
    if (!$user || !$loja || $loja->post_type !== 'lojas') {
        return false;
    }
    
    // Atualiza o meta da loja
    update_post_meta($loja_id, 'alt_lojas_usuario', $user_id);
    
    // Atualiza o autor da loja
    wp_update_post(array(
        'ID' => $loja_id,
        'post_author' => $user_id
    ));
    
    // Define o papel do usuário como lojista
    if (!in_array('administrator', $user->roles)) {
        $user->set_role('lojista');
    }
    
    return true;
}

/**
 * Cria uma loja para um usuário existente
 * 
 * @param int $user_id ID do usuário
 * @param array $dados Dados da loja
 * @return int|WP_Error ID da loja criada ou erro
 */
function alt_criar_loja_para_usuario($user_id, $dados = array()) {
    $user = get_userdata($user_id);
    if (!$user) {
        return new WP_Error('usuario_invalido', 'Usuário não encontrado');
    }
    
    // Dados padrão da loja
    $titulo = !empty($dados['titulo']) ? $dados['titulo'] : $user->display_name;
    $conteudo = !empty($dados['conteudo']) ? $dados['conteudo'] : '';
    
    // Cria o post da loja
    $loja_id = wp_insert_post(array(
        'post_title' => $titulo,
        'post_content' => $conteudo,
        'post_status' => 'publish',
        'post_type' => 'lojas',
        'post_author' => $user_id
    ));
    
    if (is_wp_error($loja_id)) {
        return $loja_id;
    }
    
    // Vincula o usuário à loja
    update_post_meta($loja_id, 'alt_lojas_usuario', $user_id);
    
    // Adiciona os meta dados da loja
    if (!empty($dados['meta'])) {
        foreach ($dados['meta'] as $key => $value) {
            update_post_meta($loja_id, $key, $value);
        }
    }
    
    // Define o papel do usuário como lojista
    if (!in_array('administrator', $user->roles)) {
        $user->set_role('lojista');
    }
    
    return $loja_id;
}

/**
 * Redireciona usuários lojistas para a edição da própria loja
 */
function alt_redirecionar_lojista_para_loja() {
    if (current_user_can('lojista')) {
        global $pagenow;
        
        if ($pagenow == 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] == 'lojas') {
            $user_id = get_current_user_id();
            $loja = alt_get_loja_do_usuario($user_id);
            
            if ($loja) {
                wp_redirect(admin_url('post.php?post=' . $loja->ID . '&action=edit'));
                exit;
            }
        }
    }
}
add_action('admin_init', 'alt_redirecionar_lojista_para_loja');

/**
 * Filtra a listagem de lojas para mostrar apenas as próprias para lojistas
 * 
 * @param WP_Query $query A consulta
 */
function alt_filtrar_lojas_por_usuario($query) {
    if (!is_admin() || !current_user_can('lojista')) {
        return;
    }
    
    if (isset($query->query['post_type']) && $query->query['post_type'] == 'lojas') {
        $query->set('author', get_current_user_id());
    }
}
add_action('pre_get_posts', 'alt_filtrar_lojas_por_usuario');

/**
 * Registra o menu e submenus para gerenciar a integração entre usuários e lojas
 */
function alt_registrar_menu_usuario_loja() {
    // Pega o usuário atual
    $user = wp_get_current_user();
    $allowed_roles = array('lojista', 'associacao');
    
    // Verifica se o usuário tem APENAS os papéis permitidos
    // Se for admin puro, não deve ver o menu
    $has_allowed_role = false;
    foreach($user->roles as $role) {
        if(in_array($role, $allowed_roles)) {
            $has_allowed_role = true;
            break;
        }
    }

    // Se não tiver os papéis permitidos, não mostra o menu
    if(!$has_allowed_role) {
        return;
    }

    // Menu principal
    add_menu_page(
        'Usuários e Lojas',
        'Usuários e Lojas', 
        'access_usuario_loja',
        'usuario_loja',
        'alt_pagina_usuario_loja',
        'dashicons-store',
        25
    );

    // Submenu "Todos os usuários e lojas"
    add_submenu_page(
        'usuario_loja',
        'Todos os usuários e lojas',
        'Todos os usuários e lojas',
        'access_usuario_loja',           // Mudado para 'read' pois já verificamos os papéis
        'usuario_loja',
        'alt_pagina_usuario_loja'
    );

    // Submenu "Adicionar novo" apenas para administradores
    if (current_user_can('associacao')) {
        add_submenu_page(
            'usuario_loja',
            'Adicionar Novo Usuário e Loja',
            'Adicionar novo usuário e loja',
            'access_usuario_loja',
            'usuario_loja_novo',
            'alt_pagina_usuario_loja_novo'
        );
    }

    // Perfil
    add_submenu_page(
        'usuario_loja',
        'Perfil',
        'Perfil',
        'access_usuario_loja',          // Mudado para 'read' pois já verificamos os papéis
        'profile.php'
    );

    add_submenu_page(
        null,
        'Editar Usuário e Loja',
        'Editar Usuário e Loja',
        'access_usuario_loja',
        'usuario_loja_editar',
        'alt_pagina_usuario_loja_editar'
    );
}
add_action('admin_menu', 'alt_registrar_menu_usuario_loja');

/**
 * Callback para a página de adicionar novo usuário e loja
 */
function alt_pagina_usuario_loja_novo() {
    // Verificar permissões
    if (!current_user_can('associacao')) {
        wp_die(__('Você não tem permissão suficiente para acessar esta página.'));
    }

    // Inclui o template do formulário
    include_once plugin_dir_path(__FILE__) . 'templates/formulario-criar-usuario-loja.php';
}

/**
 * Callback para a página de edição de usuário e loja
 */
function alt_pagina_usuario_loja_editar() {
    // Verifica se o usuário tem a capability necessária
    if (!current_user_can('lojista') && !current_user_can('associacao')) {
        wp_die(__('Você não tem permissão suficiente para acessar esta página.'));
    }
    
    // Para lojista, verifica se está editando sua própria loja
    if (current_user_can('lojista')) {
        $user_id = intval($_GET['user_id']);
        if ($user_id !== get_current_user_id()) {
            wp_die(__('Você só pode editar sua própria loja.'));
        }
    }
    
    // Verifica se os parâmetros necessários estão presentes
    if (!isset($_GET['user_id']) || !isset($_GET['loja_id'])) {
        wp_die('Parâmetros inválidos.');
    }

    $user_id = intval($_GET['user_id']);
    $loja_id = intval($_GET['loja_id']);
    $dados_edicao = alt_carregar_dados_usuario_loja($user_id, $loja_id);

    if (!$dados_edicao) {
        wp_die('Usuário ou loja não encontrados.');
    }

    include_once plugin_dir_path(__FILE__) . 'templates/formulario-editar-usuario-loja.php';
}

/**
 * Processa o formulário de criação ou edição de usuário e loja
 */
function alt_processar_usuario_loja($modo, $dados) {
    $mensagem = '';
    
    // Sanitizar dados comuns
    $nome_loja = sanitize_text_field($dados['nome_loja']);
    $loja_meta = array(
        'alt_lojas_cnpj_principal' => sanitize_text_field($dados['cnpj']),
        'alt_lojas_frase' => sanitize_text_field($dados['frase']),
        'alt_lojas_telefone_principal' => sanitize_text_field($dados['telefone']),
        'alt_lojas_whatsapp_principal' => sanitize_text_field($dados['whatsapp']),
        'alt_lojas_email_principal' => sanitize_email($dados['email_loja']),
        'alt_lojas_endereco_principal' => sanitize_text_field($dados['endereco']),
        'alt_lojas_cidade_principal' => sanitize_text_field($dados['cidade']),
        'alt_lojas_estado_principal' => sanitize_text_field($dados['estado']),
        'alt_lojas_cep' => sanitize_text_field($dados['cep']),
        'alt_lojas_google_maps' => esc_url_raw($dados['google_maps']),
        'alt_lojas_link' => esc_url_raw($dados['link_loja']),
        'alt_lojas_tipo_integracao' => sanitize_text_field($dados['tipo_integracao']),
    );

    if (isset($dados['tipo_integracao']) && $dados['tipo_integracao'] === 'json') {
        $loja_meta['alt_lojas_json_url'] = esc_url_raw($dados['json_url']);
    }

    if ($modo === 'criar_novo') {
        // Criar novo usuário e loja
        $username = sanitize_user($dados['username']);
        $email = sanitize_email($dados['email']);
        $password = $dados['password'];

        if (empty($username) || empty($email) || empty($password) || empty($nome_loja)) {
            return 'erro_campos_obrigatorios';
        }

        $usuario_id = wp_create_user($username, $password, $email);
        if (is_wp_error($usuario_id)) {
            return 'erro_criar_usuario';
        }

        alt_atualizar_user_login($usuario_id, $username, $email);

        $user = new WP_User($usuario_id);
        $user->set_role('lojista');

        $loja_data = array(
            'titulo' => $nome_loja,
            'meta' => $loja_meta,
        );

        $loja_id = alt_criar_loja_para_usuario($usuario_id, $loja_data);
        if (is_wp_error($loja_id)) {
            return 'erro_criar_loja';
        }

        // Processar upload da logo
        if (!empty($_FILES['logo']['name'])) {
            alt_processar_upload_logo($_FILES['logo'], $loja_id);
        }

        echo "<script>window.location.href = '" . admin_url('admin.php?page=usuario_loja') . "';</script>";
        exit;
    }

    if ($modo === 'editar') {
        // Editar usuário e loja existentes
        $user_id = intval($dados['user_id']);
        $loja_id = intval($dados['loja_id']);

        $dados_edicao = alt_carregar_dados_usuario_loja($user_id, $loja_id);
        if (!$dados_edicao) {
            return 'erro_id';
        }

        // Atualizar título da loja
        $loja_data = array(
            'ID' => $loja_id,
            'post_title' => $nome_loja,
        );
        $result_loja = wp_update_post($loja_data);

        if (is_wp_error($result_loja)) {
            return 'erro_atualizar_titulo';
        }

        // Atualizar meta dados da loja
        foreach ($loja_meta as $meta_key => $meta_value) {
            update_post_meta($loja_id, $meta_key, $meta_value);
        }

        // Atualizar dados do usuário
        $user_data = array(
            'ID' => $user_id,
            'display_name' => $dados['username'],
            'user_email' => sanitize_email($dados['email']),
        );

        alt_atualizar_user_login($user_id, $dados['username'], $dados['email']);
        
        $result_user = wp_update_user($user_data);
        if (is_wp_error($result_user)) {
            return 'erro_atualizar_usuario';
        }

        // Atualizar senha do usuário, se fornecida
        if (!empty($dados['password'])) {
            wp_update_user(array(
                'ID' => $user_id,
                'user_pass' => $dados['password']
            ));
        }

        // Processar upload da logo
        if (!empty($_FILES['logo']['name'])) {
            alt_processar_upload_logo($_FILES['logo'], $loja_id);
        }
        
        echo "<script>window.location.href = '" . admin_url('admin.php?page=usuario_loja') . "';</script>";
        exit;
    }

    echo "<script>window.location.href = '" . admin_url('admin.php?page=usuario_loja') . "';</script>";
    exit;
}

/**
 * Callback para a página de gerenciamento de usuários e lojas
 */
function alt_pagina_usuario_loja() {
    // Verifica se o usuário tem um dos papéis permitidos
    if (!current_user_can('lojista') && !current_user_can('associacao')) {
        wp_die(__('Você não tem permissão suficiente para acessar esta página.'));
    }

    // Inicializar variáveis
    $editing = false;
    $dados_edicao = false;
    $user_id = 0;
    $user_edit_id = 0;
    $loja_id = 0;
    $loja_edit_id = 0;

    // Construir argumentos da consulta
    $args = array(
        'post_type' => 'lojas',
        'posts_per_page' => -1,
        'orderby' => 'date',
        'order' => 'DESC',
        'meta_query' => array(
            array(
                'key' => 'alt_lojas_usuario',
                'compare' => 'EXISTS',
            )
        )
    );

    // Filtro por nome da loja
    if (!empty($_GET['filtro_loja'])) {
        $args['s'] = sanitize_text_field($_GET['filtro_loja']);
    }

    // Filtro por período
    if (!empty($_GET['filtro_periodo'])) {
        list($ano, $mes) = explode('-', $_GET['filtro_periodo']);
        $args['date_query'] = array(
            array(
                'year' => $ano,
                'month' => $mes
            )
        );
    }

    // Buscar lojas
    $lojas_vinculadas = get_posts($args);

    // Filtro por nome do usuário (pós-query)
    if (!empty($_GET['filtro_usuario'])) {
        $lojas_filtradas = array();
        $termo_busca = strtolower(sanitize_text_field($_GET['filtro_usuario']));
        
        foreach ($lojas_vinculadas as $loja) {
            $user_id = get_post_meta($loja->ID, 'alt_lojas_usuario', true);
            $user = get_userdata($user_id);
            
            if ($user && (
                strpos(strtolower($user->display_name), $termo_busca) !== false ||
                strpos(strtolower($user->user_login), $termo_busca) !== false
            )) {
                $lojas_filtradas[] = $loja;
            }
        }
        
        $lojas_vinculadas = $lojas_filtradas;
    }

    // Verificar se estamos no modo de edição
    if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['user_id']) && isset($_GET['loja_id'])) {
        $user_id = intval($_GET['user_id']);
        $loja_id = intval($_GET['loja_id']);
        $dados_edicao = alt_carregar_dados_usuario_loja($user_id, $loja_id);

        if ($dados_edicao) {
            $editing = true;
            $user_edit_id = $dados_edicao['user']->ID;
            $loja_edit_id = $dados_edicao['loja_meta']['loja_id'];
        } else {
            wp_redirect(menu_page_url('usuario_loja', false));
            exit;
        }
    }

    // Processamento do formulário
    if (isset($_POST['submit_usuario_loja'])) {
        if (isset($_POST['criar_usuario_loja_nonce']) && wp_verify_nonce($_POST['criar_usuario_loja_nonce'], 'usuario_loja')) {
            $modo = 'criar_novo';
        } elseif (isset($_POST['editar_usuario_loja_nonce']) && wp_verify_nonce($_POST['editar_usuario_loja_nonce'], 'usuario_loja')) {
            $modo = 'editar';
        } else {
            wp_die('Verificação de segurança falhou. Tente novamente.');
        }

        $mensagem = alt_processar_usuario_loja($modo, $_POST);
        // Redirecionar para a listagem com mensagem
        // Gera a URL base sem os parâmetros indesejados
        $url_base = admin_url('admin.php?page=usuario_loja');

        // Remove os parâmetros indesejados da URL
        $url_redirecionamento = remove_query_arg(array('action', 'user_id', 'loja_id'), $url_base);

        // Adiciona o parâmetro de mensagem
        $url_redirecionamento = add_query_arg('message', $mensagem, $url_redirecionamento);

        // Redireciona para a URL final
        header('Location: ' . esc_url_raw($url_redirecionamento));
        exit;
    }

    // Exibir a página
    ?>
    <div class="wrap">
        <h1 class="wp-heading-inline"><?php echo esc_html(get_admin_page_title()); ?></h1>
            <?php if (current_user_can('associacao')): ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=usuario_loja_novo')); ?>" class="page-title-action">Adicionar Novo Usuário e Loja</a>
                <hr class="wp-header-end">
                
                <a href="<?php echo esc_url(admin_url('admin-post.php?action=exportar_planilha')); ?>" class="button button-primary">
                    Exportar Dados para Planilha
                </a>
            <?php endif; ?>

        <!-- Formulário de Filtros -->
        <form method="get" action="">
            <input type="hidden" name="page" value="usuario_loja">
            
            <!-- Campos de filtro -->
            <div style="display: flex; align-items: center; gap: 20px; flex-wrap: wrap; margin-bottom: 20px; padding-top: 10px;">
                <div>
                    <label for="filtro_usuario">Nome do Usuário</label><br>
                    <input type="text" name="filtro_usuario" id="filtro_usuario" value="<?php echo esc_attr($_GET['filtro_usuario'] ?? ''); ?>" class="regular-text">
                </div>

                <div>
                    <label for="filtro_loja">Nome da Loja</label><br>
                    <input type="text" name="filtro_loja" id="filtro_loja" value="<?php echo esc_attr($_GET['filtro_loja'] ?? ''); ?>" class="regular-text">
                </div>

                <div>
                    <label for="filtro_periodo">Período</label><br>
                    <select name="filtro_periodo" id="filtro_periodo" style="width: 200px;">
                        <option value="">Todas as datas</option>
                        <?php
                        $datas = alt_get_datas_disponiveis();
                        $periodo_selecionado = $_GET['filtro_periodo'] ?? '';
                        
                        foreach ($datas as $data) {
                            printf(
                                '<option value="%s" %s>%s</option>',
                                esc_attr($data['valor']),
                                selected($periodo_selecionado, $data['valor'], false),
                                esc_html($data['label'])
                            );
                        }
                        ?>
                    </select>
                </div>
            </div>

            <!-- Botões -->
            <div style="margin-bottom: 20px;">
                <input type="submit" class="button button-secondary" value="Buscar">
                <a href="<?php echo esc_url(admin_url('admin.php?page=usuario_loja')); ?>" class="button">Limpar Filtros</a>
            </div>
        </form>
        
        <?php
        // Exibir mensagens
        if (isset($_GET['message'])) {
            $tipo = 'notice-info';
            $mensagem = '';

            switch ($_GET['message']) {
                case 'sucesso_criar':
                    $tipo = 'notice-success';
                    $mensagem = 'Usuário e loja criados com sucesso!';
                    break;
                case 'sucesso_atualizar':
                    $tipo = 'notice-success';
                    $mensagem = 'Usuário e loja atualizados com sucesso!';
                    break;
                case 'erro_campos_obrigatorios':
                    $tipo = 'notice-error';
                    $mensagem = 'Preencha todos os campos obrigatórios.';
                    break;
                case 'erro_criar_usuario':
                    $tipo = 'notice-error';
                    $mensagem = 'Erro ao criar usuário.';
                    break;
                case 'erro_criar_loja':
                    $tipo = 'notice-error';
                    $mensagem = 'Erro ao criar loja.';
                    break;
                case 'erro_atualizar_titulo':
                    $tipo = 'notice-error';
                    $mensagem = 'Erro ao atualizar o título da loja.';
                    break;
                case 'sucesso_exclusao':
                    $tipo = 'notice-success';
                    $mensagem = 'Loja e usuário excluídos com sucesso.';
                    break;
                case 'erro_exclusao':
                    $tipo = 'notice-error';
                    $mensagem = 'Erro ao excluir alguns itens.';
                    break;
            }

            if ($mensagem) {
                echo '<div class="notice ' . $tipo . ' is-dismissible"><p>' . $mensagem . '</p></div>';
            }
        }
        ?>

        <!-- Conteúdo das abas -->
        <div id="vinculados" class="tab-content">
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="form-vinculos">
                <input type="hidden" name="action" value="excluir_vinculos">
                <?php wp_nonce_field('excluir_vinculos', 'excluir_vinculos_nonce'); ?>
                
                <div class="tablenav top">
                    <div class="alignleft actions bulkactions">
                        <select name="acao">
                            <option value="-1">Ações em massa</option>
                            <option value="excluir">Excluir</option>
                        </select>
                        <input type="submit" class="button action" value="Aplicar">
                    </div>
                </div>

                <table class="wp-list-table widefat fixed striped users">
                    <thead>
                        <tr>
                            <th class="check-column">
                                <input type="checkbox" id="cb-select-all-1">
                            </th>
                            <th>Nome de Usuário</th>
                            <th>Nome da Loja</th>
                            <th>E-mail</th>
                            <th>Data</th>
                            <th>Veículos</th>
                            <th>Status</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($lojas_vinculadas)): ?>
                            <tr><td colspan="6">Nenhuma vinculação encontrada.</td></tr>
                        <?php else: ?>
                            <?php foreach ($lojas_vinculadas as $loja): ?>
                                <?php
                                $user_id = get_post_meta($loja->ID, 'alt_lojas_usuario', true);
                                $user = get_userdata($user_id);
                                $status = alt_get_status_loja($loja->ID);
                                $veiculos = alt_get_qtd_veiculos_loja($loja->ID) ?? 0;
                                $status_texto = $status;
                                $data_inatividade = alt_data_inatividade_loja($loja->ID);
                                if (isset($data_inatividade)) {
                                    $status_texto .= " (" . $data_inatividade . ")"; 
                                }
                                if ($user):
                                ?>
                                <tr class="<?= $status == 'Inativa' ? 'loja_inativa' : ''; ?>">                              
                                    <th scope="row" class="check-column">
                                        <input type="checkbox" name="vinculos[]" value="<?php echo esc_attr($loja->ID . '|' . $user_id); ?>">
                                    </th>
                                    <td><span><?php echo esc_html($user->display_name); ?></span></td>
                                    <td><span><?php echo esc_html($loja->post_title); ?></span></td>
                                    <td><span><?php echo esc_html($user->user_email); ?></span></td>
                                    <td><span><?php echo esc_html(get_the_date(get_option('date_format'), $loja->ID)); ?></span></td>
                                    <td><span><?php echo esc_html($veiculos); ?></span></td>
                                    <td><span><?php echo esc_html($status_texto); ?></span></td>
                                    <td>
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=usuario_loja_editar&user_id=' . $user->ID . '&loja_id=' . $loja->ID)); ?>" class="button">Editar</a>
                                    </td>
                                </tr>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </form>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Preview da logo
            function readURL(input) {
                if (input.files && input.files[0]) {
                    var reader = new FileReader();
                    reader.onload = function(e) {
                        $('#logo-preview').attr('src', e.target.result);
                    }
                    reader.readAsDataURL(input.files[0]);
                }
            }
            $("#criar_logo").change(function() {
                readURL(this);
            });

            // Toggle do campo JSON URL
            function toggleJsonUrl() {
                if ($("#criar_tipo_integracao").val() === "json") {
                    $("#json_url_row").show();
                } else {
                    $("#json_url_row").hide();
                }
            }
            
            toggleJsonUrl();
            $("#criar_tipo_integracao").on("change", toggleJsonUrl);
        });
        </script>
    </div>
    <?php
}

/**
 * Processa o formulário de vinculação
 */
function alt_processar_vinculacao_usuario_loja() {
    if (isset($_POST['vincular_usuario_loja']) && isset($_POST['vincular_usuario_loja_nonce'])) {
        if (!wp_verify_nonce($_POST['vincular_usuario_loja_nonce'], 'vincular_usuario_loja')) {
            wp_die('Verificação de segurança falhou. Tente novamente.');
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $loja_id = isset($_POST['loja_id']) ? intval($_POST['loja_id']) : 0;
        
        if ($user_id && $loja_id) {
            $resultado = alt_vincular_usuario_loja($user_id, $loja_id);
            
            if ($resultado) {
                wp_redirect(add_query_arg('vinculado', 1, menu_page_url('usuario_loja', false)));
                exit;
            } else {
                wp_redirect(add_query_arg('erro', 1, menu_page_url('usuario_loja', false)));
                exit;
            }
        }
    }
}
add_action('admin_init', 'alt_processar_vinculacao_usuario_loja');

/**
 * Carrega os dados do usuário e loja para edição
 * 
 * @param int $user_id ID do usuário
 * @param int $loja_id ID da loja
 * @return array|false Array com chaves 'user' e 'loja' ou false se inválido
 */
function alt_carregar_dados_usuario_loja($user_id, $loja_id) {
    // Validar IDs
    $user_id = intval($user_id);
    $loja_id = intval($loja_id);
    
    if (!$user_id || !$loja_id) {
        return false;
    }
    
    // Carregar dados
    $user = get_userdata($user_id);
    $loja = get_post($loja_id);
    
    // Verificar se o usuário e loja existem
    if (!$user || !$loja || $loja->post_type !== 'lojas') {
        return false;
    }
    
    // Verificar se o usuário está realmente vinculado à loja
    $loja_usuario_id = get_post_meta($loja_id, 'alt_lojas_usuario', true);
    
    if ($loja_usuario_id != $user_id) {
        return false;
    }

    // Carregar metadados da loja para facilitar o acesso
    $loja_meta = array(
        'loja_id' => $loja_id,
        'cnpj' => get_post_meta($loja_id, 'alt_lojas_cnpj_principal', true),
        'frase' => get_post_meta($loja_id, 'alt_lojas_frase', true),
        'telefone' => get_post_meta($loja_id, 'alt_lojas_telefone_principal', true),
        'whatsapp' => get_post_meta($loja_id, 'alt_lojas_whatsapp_principal', true),
        'email' => get_post_meta($loja_id, 'alt_lojas_email_principal', true),
        'endereco' => get_post_meta($loja_id, 'alt_lojas_endereco_principal', true),
        'cidade' => get_post_meta($loja_id, 'alt_lojas_cidade_principal', true),
        'estado' => get_post_meta($loja_id, 'alt_lojas_estado_principal', true),
        'cep' => get_post_meta($loja_id, 'alt_lojas_cep', true),
        'google_maps' => get_post_meta($loja_id, 'alt_lojas_google_maps', true),
        'link' => get_post_meta($loja_id, 'alt_lojas_link', true),
        'tipo_integracao' => get_post_meta($loja_id, 'alt_lojas_tipo_integracao', true),
        'json_url' => get_post_meta($loja_id, 'alt_lojas_json_url', true)
    );
    
    // Remover hook temporariamente (remover em produção)
    remove_action('save_post', 'alt_sincronizar_loja_usuario', 20, 2);
    
    // Retornar os dados
    return array(
        'user' => $user,
        'loja' => $loja,
        'loja_meta' => $loja_meta
    );
}

/**
 * Exporta dados de usuários e lojas para um arquivo CSV
 */
function alt_exportar_csv() {
    // Verifique permissões
    if (!current_user_can('manage_options') && !current_user_can('associacao')) {
        wp_die('Você não tem permissão para acessar esta funcionalidade.');
    }

    // Defina os cabeçalhos para download do CSV
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=usuarios_lojas.csv');

    // Abra a saída para escrever o CSV
    $output = fopen('php://output', 'w');

    // Adicione os cabeçalhos das colunas
    fputcsv($output, array(
        'Nome do Usuário',
        'E-mail do Usuário',
        'Usuário na Integração',
        'Nome da Loja',
        'CNPJ',
        'Frase/Slogan',
        'Telefone',
        'WhatsApp',
        'E-mail da Loja',
        'Endereço',
        'Cidade',
        'Estado',
        'CEP',
        'Link Google Maps',
        'Link da Loja',
        'Tipo de Integração',
        'URL JSON'
    ));

    // Obtenha os dados de usuários e lojas
    $args = array(
        'post_type' => 'lojas',
        'posts_per_page' => -1,
        'meta_query' => array(
            array(
                'key' => 'alt_lojas_usuario',
                'compare' => 'EXISTS',
            ),
        ),
    );

    $lojas = get_posts($args);

    // Adicione os dados ao CSV
    foreach ($lojas as $loja) {
        $user_id = get_post_meta($loja->ID, 'alt_lojas_usuario', true);
        $user = get_userdata($user_id);

        if ($user) {
            fputcsv($output, array(
                $user->display_name,
                $user->user_email,
                $user->user_login,
                $loja->post_title,
                get_post_meta($loja->ID, 'alt_lojas_cnpj_principal', true),
                get_post_meta($loja->ID, 'alt_lojas_frase', true),
                get_post_meta($loja->ID, 'alt_lojas_telefone_principal', true),
                get_post_meta($loja->ID, 'alt_lojas_whatsapp_principal', true),
                get_post_meta($loja->ID, 'alt_lojas_email_principal', true),
                get_post_meta($loja->ID, 'alt_lojas_endereco_principal', true),
                get_post_meta($loja->ID, 'alt_lojas_cidade_principal', true),
                get_post_meta($loja->ID, 'alt_lojas_estado_principal', true),
                get_post_meta($loja->ID, 'alt_lojas_cep', true),
                get_post_meta($loja->ID, 'alt_lojas_google_maps', true),
                get_post_meta($loja->ID, 'alt_lojas_link', true),
                get_post_meta($loja->ID, 'alt_lojas_tipo_integracao', true),
                get_post_meta($loja->ID, 'alt_lojas_json_url', true),
            ));
        }
    }

    // Feche o arquivo de saída
    fclose($output);
    exit;
}

add_action('admin_post_exportar_planilha', 'alt_exportar_csv');

/**
 * Busca as datas distintas disponíveis
 */
function alt_get_datas_disponiveis() {
    global $wpdb;
    $query = "
        SELECT DISTINCT 
            YEAR(post_date) as ano,
            MONTH(post_date) as mes
        FROM {$wpdb->posts}
        WHERE post_type = 'lojas'
        AND post_status = 'publish'
        ORDER BY ano DESC, mes DESC
    ";
    
    $resultados = $wpdb->get_results($query);
    $datas = array();
    
    foreach ($resultados as $data) {
        $datas[] = array(
            'valor' => $data->ano . '-' . str_pad($data->mes, 2, '0', STR_PAD_LEFT),
            'label' => date_i18n('F Y', strtotime($data->ano . '-' . $data->mes . '-01'))
        );
    }
    
    return $datas;
}

/**
 * Exclui vínculos de usuários e lojas
 */
function alt_excluir_vinculos() {
    if (!current_user_can('associacao')) {
        wp_die('Você não tem permissão para realizar esta ação.');
    }

    if (!isset($_POST['vinculos']) || !is_array($_POST['vinculos'])) {
        return;
    }

    $mensagens = array();
    foreach ($_POST['vinculos'] as $vinculo) {
        list($loja_id, $user_id) = explode('|', $vinculo);
        
        // Exclui a loja
        $resultado_loja = wp_delete_post($loja_id, true);
        
        // Exclui o usuário
        if ($resultado_loja) {
            $resultado_user = wp_delete_user($user_id);
            if ($resultado_user) {
                $mensagens[] = 'sucesso';
            } else {
                $mensagens[] = 'erro_usuario';
            }
        } else {
            $mensagens[] = 'erro_loja';
        }
    }

    // Redireciona com mensagem apropriada
    $tipo_mensagem = in_array('erro_loja', $mensagens) || in_array('erro_usuario', $mensagens) 
        ? 'erro_exclusao' 
        : 'sucesso_exclusao';

    wp_redirect(add_query_arg('message', $tipo_mensagem, admin_url('admin.php?page=usuario_loja')));
    exit;
}
add_action('admin_post_excluir_vinculos', 'alt_excluir_vinculos');

// Adicione esta função junto às outras funções do arquivo
function alt_admin_enqueue_scripts($hook) {
    // Corrige verificação do hook para incluir ambas páginas
    if (!in_array($hook, array(
        'toplevel_page_usuario_loja',
        'usuarios-e-lojas_page_usuario_loja_novo',
        'usuarios-e-lojas_page_usuario_loja_editar'
    ))) {
        return;
    }
    
    wp_register_script('usuario-loja-admin', false, array('jquery'));
    wp_enqueue_script('usuario-loja-admin');
    
    // Script inline que funciona tanto para criação quanto para edição
    wp_add_inline_script('usuario-loja-admin', '
        jQuery(document).ready(function($) {
            // Toggle do campo JSON URL
            function toggleJsonUrl() {
                var tipoIntegracao = $("#criar_tipo_integracao, #editar_tipo_integracao").val();
                if (tipoIntegracao === "json") {
                    $("#json_url_row").show();
                } else {
                    $("#json_url_row").hide();
                }
            }
            
            // Executa no carregamento
            toggleJsonUrl();
            
            // Monitora ambos selects (criar e editar)
            $("#criar_tipo_integracao, #editar_tipo_integracao").on("change", toggleJsonUrl);
            
            // Preview da logo para ambos forms
            function readURL(input) {
                if (input.files && input.files[0]) {
                    var reader = new FileReader();
                    reader.onload = function(e) {
                        $("#logo-preview").attr("src", e.target.result);
                    }
                    reader.readAsDataURL(input.files[0]);
                }
            }
            
            $("#criar_logo, #editar_logo").change(function() {
                readURL(this);
            });
        });
    ');
}
add_action('admin_enqueue_scripts', 'alt_admin_enqueue_scripts');

/**
 * Retorna a lista de estados brasileiros
 * 
 * @return array Lista de estados
 */
function alt_get_lista_estados() {
    return array(
        'AC' => 'Acre',
        'AL' => 'Alagoas',
        'AP' => 'Amapá',
        'AM' => 'Amazonas',
        'BA' => 'Bahia',
        'CE' => 'Ceará',
        'DF' => 'Distrito Federal',
        'ES' => 'Espírito Santo',
        'GO' => 'Goiás',
        'MA' => 'Maranhão',
        'MT' => 'Mato Grosso',
        'MS' => 'Mato Grosso do Sul',
        'MG' => 'Minas Gerais',
        'PA' => 'Pará',
        'PB' => 'Paraíba',
        'PR' => 'Paraná',
        'PE' => 'Pernambuco',
        'PI' => 'Piauí',
        'RJ' => 'Rio de Janeiro',
        'RN' => 'Rio Grande do Norte',
        'RS' => 'Rio Grande do Sul',
        'RO' => 'Rondônia',
        'RR' => 'Roraima',
        'SC' => 'Santa Catarina',
        'SP' => 'São Paulo',
        'SE' => 'Sergipe',
    );
}

function alt_atualizar_user_login($usuario_id, $username, $email) {
    // Formata o nome de usuário para user_login (sem acentos, sem espaços, sem caracteres especiais)
    $user_login = remove_accents($username); // Remove acentos
    $user_login = preg_replace('/[^a-zA-Z0-9]/', '', $user_login); // Remove caracteres especiais e espaços
    $user_login = strtolower($user_login); // Converte para minúsculas
    
    // Se após a limpeza o login estiver vazio, use uma parte do email
    if (empty($user_login)) {
        $user_login = preg_replace('/[@.]/', '', substr($email, 0, strpos($email, '@')));
    }

    global $wpdb;
    
    $login_existe = $wpdb->get_var($wpdb->prepare(
        "SELECT ID FROM {$wpdb->users} WHERE user_login = %s AND ID != %d",
        $user_login,
        $usuario_id
    ));

    if (!$login_existe) {
        $wpdb->update(
            $wpdb->users,
            array('user_login' => $user_login),
            array('ID' => $usuario_id)
        );
    }
}

/**
 * Gera um link para excluir uma loja e seu usuário vinculado permanentemente
 * 
 * @param int $loja_id ID da loja a ser excluída
 * @param bool $force_delete Se true, exclui permanentemente (padrão true)
 * @return string URL de exclusão
 */
function alt_get_delete_loja_usuario_link($loja_id, $force_delete = true) {
    if (empty($loja_id) || !is_numeric($loja_id)) {
        return '';
    }
    
    // Ação sempre será exclusão permanente
    $action = 'delete_loja_usuario';
    
    // Cria o nonce de segurança
    $nonce = wp_create_nonce($action . '_' . $loja_id);
    
    // Monta a URL de exclusão
    $delete_url = admin_url(sprintf(
        'admin-post.php?action=%s&loja_id=%d&_wpnonce=%s',
        $action,
        $loja_id,
        $nonce
    ));
    
    return $delete_url;
}

/**
 * Processa a exclusão permanente de loja e usuário vinculado
 */
function alt_delete_loja_usuario() {
    // Verificação de segurança
    if (!isset($_GET['loja_id']) || !isset($_GET['_wpnonce'])) {
        wp_die('Parâmetros inválidos.');
    }

    $loja_id = intval($_GET['loja_id']);
    
    // Verifica o nonce
    if (!wp_verify_nonce($_GET['_wpnonce'], 'delete_loja_usuario_' . $loja_id)) {
        wp_die('Verificação de segurança falhou.');
    }
    
    // Verifica permissões
    if (!current_user_can('associacao')) {
        wp_die('Você não tem permissão para realizar esta ação.');
    }
    
    // Obtém o usuário vinculado à loja
    $user_id = get_post_meta($loja_id, 'alt_lojas_usuario', true);
    
    // Exclui a loja permanentemente
    wp_delete_post($loja_id, true); // true = força exclusão permanente
    
    // Exclui o usuário permanentemente (sem atribuir posts a outro usuário)
    if ($user_id) {
        require_once(ABSPATH . 'wp-admin/includes/user.php');
        wp_delete_user($user_id);
    }
    
    // Redireciona para a página de listagem
    wp_redirect(admin_url('admin.php?page=usuario_loja&message=sucesso_exclusao'));
    exit;
}
add_action('admin_post_delete_loja_usuario', 'alt_delete_loja_usuario');

/**
 * Inativa uma loja
 */
function alt_inativar_loja() {
    // Verificação de segurança
    if (!isset($_GET['loja_id']) || !isset($_GET['_wpnonce'])) {
        wp_die('Parâmetros inválidos.');
    }

    $loja_id = intval($_GET['loja_id']);
    
    // Verifica o nonce
    if (!wp_verify_nonce($_GET['_wpnonce'], 'inativar_loja_' . $loja_id)) {
        wp_die('Verificação de segurança falhou.');
    }
    
    // Verifica permissões
    if (!current_user_can('associacao') && 
        !(current_user_can('lojista') && alt_lojista_pode_editar($loja_id))) {
        wp_die('Você não tem permissão para realizar esta ação.');
    }
    
    // Salva status de inativo e data atual
    update_post_meta($loja_id, 'alt_lojas_status_loja', 'inativa');
    update_post_meta($loja_id, 'alt_lojas_data_inatividade', date('Y-m-d'));
    
    // Redireciona de volta para a página de edição
    wp_redirect(add_query_arg(
        array(
            'message' => 'loja_inativada',
            'user_id' => get_post_meta($loja_id, 'alt_lojas_usuario', true),
            'loja_id' => $loja_id
        ), 
        admin_url('admin.php?page=usuario_loja_editar')
    ));
    exit;
}
add_action('admin_post_inativar_loja', 'alt_inativar_loja');

/**
 * Reativa uma loja
 */
function alt_reativar_loja() {
    // Verificação de segurança
    if (!isset($_GET['loja_id']) || !isset($_GET['_wpnonce'])) {
        wp_die('Parâmetros inválidos.');
    }

    $loja_id = intval($_GET['loja_id']);
    
    // Verifica o nonce
    if (!wp_verify_nonce($_GET['_wpnonce'], 'reativar_loja_' . $loja_id)) {
        wp_die('Verificação de segurança falhou.');
    }
    
    // Verifica permissões
    if (!current_user_can('associacao') && 
        !(current_user_can('lojista') && alt_lojista_pode_editar($loja_id))) {
        wp_die('Você não tem permissão para realizar esta ação.');
    }
    
    // Remove o status e a data de inatividade
    delete_post_meta($loja_id, 'alt_lojas_status_loja');
    delete_post_meta($loja_id, 'alt_lojas_data_inatividade');
    
    // Redireciona de volta para a página de edição
    wp_redirect(add_query_arg(
        array(
            'message' => 'loja_reativada',
            'user_id' => get_post_meta($loja_id, 'alt_lojas_usuario', true),
            'loja_id' => $loja_id
        ), 
        admin_url('admin.php?page=usuario_loja_editar')
    ));
    exit;
}
add_action('admin_post_reativar_loja', 'alt_reativar_loja');

/**
 * Verifica se um lojista pode editar uma determinada loja
 * 
 * @param int $loja_id ID da loja
 * @return bool Verdadeiro se o lojista atual pode editar a loja
 */
function alt_lojista_pode_editar($loja_id) {
    $loja_usuario_id = get_post_meta($loja_id, 'alt_lojas_usuario', true);
    return $loja_usuario_id == get_current_user_id();
}

/**
 * Retorna os IDs dos usuários vinculados a lojas inativas
 * 
 * @return array IDs dos usuários das lojas inativas
 */
function alt_get_usuarios_lojas_inativas() {
    global $wpdb;
    
    // Consulta única que busca diretamente os usuários de lojas inativas
    $query = "
        SELECT pm.meta_value AS user_id
        FROM {$wpdb->postmeta} pm
        JOIN {$wpdb->postmeta} pm_status ON pm.post_id = pm_status.post_id
        WHERE pm.meta_key = 'alt_lojas_usuario'
        AND pm_status.meta_key = 'alt_lojas_status_loja'
        AND pm_status.meta_value = 'inativa'
    ";
    
    return $wpdb->get_col($query);
}